/*
 # (C) 2009-2010 Jürgen Löb
 #
 # This program is distributed in the hope that it will be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 # See the Attribution-NonCommercial-ShareAlike 3.0 Unported for more details.
 #
 */
#include <osg/Switch>
#include <osgText/Text>
#include <osgGA/GUIEventHandler>

#include <osgViewer/Viewer>
#include <osgViewer/GraphicsWindow>
#include <osg/Geometry>
#include <osg/Image>
#include <osg/BufferObject>
#include <osgGA/TrackballManipulator>
#include <osgGA/FlightManipulator>
#include <osgGA/DriveManipulator>
#include <osgGA/KeySwitchMatrixManipulator>
#include <osgGA/StateSetManipulator>
#include <osgGA/AnimationPathManipulator>
#include <osgGA/TerrainManipulator>
#include <osg/TexMat>
#include <osgViewer/ViewerEventHandlers>
#include <osgViewer/GraphicsWindow>
#include <osg/Texture2D>
#include <osgUtil/SceneView>
#include <osgUtil/Statistics>
#include <osgDB/ReadFile>
#include <osg/Shader>
#include <osgUtil/Optimizer>
#include <osg/ApplicationUsage>
#include <iostream>
#include <stdio.h>
#include <string.h>

#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>
#include <osgGA/GUIEventAdapter>
#include <string>
#include <sstream>
#include <limits.h>

#include "osd.hpp"

#include "VR920TrackingManipulator.hpp"
#include "ZoomManipulator.hpp"

enum Filter {
	BW = 1, SEPIA = 2, NONE = 0
};

std::string vsrc = "void main(void)"
	"{"
	"  gl_Position = ftransform();"
	"  gl_TexCoord[0] = gl_MultiTexCoord0;"
	"}";
std::string
		fsrc =
				"uniform sampler2D sceneBuffer;"
					"uniform float gamma;"
					"uniform float brightness;"
					"uniform float contrast;"
					"uniform vec3 red;"
					"uniform vec3 green;"
					"uniform vec3 blue;"
					"void main(void)"
					"{"
					"  vec2 uv = gl_TexCoord[0].xy;"
					"  vec3 color_in = texture2D(sceneBuffer, uv).rgb;"
					"  vec3 color={color_in.r*red.r+color_in.g*red.g+color_in.b*red.b,color_in.r*green.r+color_in.g*green.g+color_in.b*green.b,color_in.r*blue.r+color_in.g*blue.g+color_in.b*blue.b};"
					"  color+=brightness;"
					"  color=(color-0.5)*contrast+0.5;"
					"  gl_FragColor.rgb = pow(color, vec3(1.0 / gamma));"
					"  gl_FragColor.a = 1.0;"
					"}";

std::vector<std::string>::size_type filenum = 0;
typedef std::vector<std::string> FileList;
FileList fileList;
bool slideshow = false;
bool vr920 = false;
bool shader = true;
int filtermode = NONE;
float contrast = 1.0f;
float brightness = 0.0f;
float gammaval = 1.0f;
float distance = 1.0f;
float slideDelay = 15.0f;
float parallax = 0.0f;

float _fovy = 1.0f;
float _radius = 1.0f;
float _height = 2* _radius * tan(_fovy * 0.5f);
float _length = osg::PI * _radius; // half a cylinder.
bool newImage = false;

osg::ref_ptr<osg::Image> imageLeft = new osg::Image();
osg::ref_ptr<osg::Image> imageRight = new osg::Image();
osg::ref_ptr<osg::Texture2D> textureLeft = new osg::Texture2D;
osg::ref_ptr<osg::Texture2D> textureRight = new osg::Texture2D;
osg::Switch* root = new osg::Switch();
osg::ref_ptr<OSD> osd;
osg::ref_ptr<osg::Switch> rootNode;
osgUtil::Optimizer optimizer;
osg::Program* pgm;
osg::ref_ptr<osgGA::KeySwitchMatrixManipulator> keyswitchManipulator;

std::string multicastAddress = "224.0.0.42";
int port = 4242;

char msgString[200];

class myKeyboardEventHandler: public osgGA::GUIEventHandler {
public:
	virtual bool handle(const osgGA::GUIEventAdapter& ea,
			osgGA::GUIActionAdapter&);
	virtual void accept(osgGA::GUIEventHandlerVisitor& v) {
		v.visit(*this);
	}
	;
};

// read image and split into two
osgDB::ReaderWriter::ReadResult readJPGImages(const std::string& file,
		osg::Image* left, osg::Image* right) {
	std::string ext = osgDB::getFileExtension(file);
	if (ext.find("jpeg") == ext.npos && ext.find("jpg") == ext.npos
			&& ext.find("JPG") == ext.npos && ext.find("jps") == ext.npos
			&& ext.find("JPS") == ext.npos) {
		osd->setText("No Image loaded.");
		return osgDB::ReaderWriter::ReadResult::FILE_NOT_HANDLED;
	}
	//read image file

	std::ifstream filePath(file.c_str(), std::ios::in | std::ios::binary);

	if (!filePath) {
		osd->setText("No Image loaded.");
		return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}

	// try jp2 (jasper) since I heard it is better
	osgDB::ReaderWriter* rw;
	rw = osgDB::Registry::instance()->getReaderWriterForExtension("jp2");
	if (!rw) {
		rw = osgDB::Registry::instance()->getReaderWriterForExtension("jpg");
		std::cout << "No jp2 plugin found! Falling back to jp2 plugin."
				<< std::endl;
	}
	// fallback to jpeg
	if (!rw) {
		osd->setText("No jpg plugin found!");
		std::cout << "No jpg plugin found!!!" << std::endl;
		return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}
	std::cout << rw->className() << std::endl;

	osgDB::ReaderWriter::ReadResult rs = rw->readImage(filePath, 0);

	//check image
	if (!rs.validImage()) {
		osd->setText("Error loading Image!");
		std::cout << "Error loading Image: " << file.c_str() << "!"
				<< std::endl;
		std::cout << rs.error() << std::endl;
		//return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}

	// split image
	osg::Image* image = rs.takeImage();
	//image->scaleImage(image->s()/8*8,image->t()/4*4,image->r());
	int internalFormat = image->getInternalTextureFormat();
	if (internalFormat > 4)
		internalFormat = 3; // is internalFormat broken in ReaderWriterJPEG?
	int s = image->s();
	int t = image->t();
	unsigned char* data;
	unsigned char* datal = new unsigned char[internalFormat * s * t / 2];
	unsigned char* datar = new unsigned char[internalFormat * s * t / 2];

	unsigned int pixelFormat = internalFormat == 1 ? GL_LUMINANCE
			: internalFormat == 2 ? GL_LUMINANCE_ALPHA
					: internalFormat == 3 ? GL_RGB
							: internalFormat == 4 ? GL_RGBA : (GLenum) -1;

	unsigned int dataType = GL_UNSIGNED_BYTE;
	data = image->data();
	for (int rows = 0; rows < t; rows++) {
		memcpy(datal + internalFormat * rows * s / 2, data + internalFormat
				* (rows * s), internalFormat * s / 2);
		memcpy(datar + internalFormat * rows * s / 2, data + internalFormat
				* (rows * s + s / 2), internalFormat * s / 2);
	}

	//apply
	textureLeft->setTextureSize(s / 2, t);
	textureRight->setTextureSize(s / 2, t);

	left->setImage(s / 2, t, 1, internalFormat, pixelFormat, dataType, datal,
			osg::Image::USE_NEW_DELETE);

	right->setImage(s / 2, t, 1, internalFormat, pixelFormat, dataType, datar,
			osg::Image::USE_NEW_DELETE);

	if (data)
		delete data;
	textureLeft->dirtyTextureObject();
	textureRight->dirtyTextureObject();
	filePath.close();
	return osgDB::ReaderWriter::ReadResult::FILE_LOADED;
}

int findPattern(std::ifstream* stream, const char* pattern, int offset,
		int patternsize) {
	stream->seekg(0, std::ios_base::end);
	const std::streamsize sz = stream->tellg();
	char* data = new char[sz - offset];
	stream->seekg(offset, std::ios_base::beg);
	stream->read(data, sz);
	for (int i = 0; i < sz - offset; i += 1) {
		if (memcmp(pattern, data + i, patternsize) == 0) {
			std::cout << "MPO offset: " << i + offset << std::endl;
			delete data;
			return i + offset;
		}
	}
	delete data;

	return 0; //4888064;
}

osgDB::ReaderWriter::ReadResult readMPOImages(const std::string& file,
		osg::Image* left, osg::Image* right) {
	std::string ext = osgDB::getFileExtension(file);
	if (ext.find("mpo") == ext.npos && ext.find("MPO") == ext.npos) {
		osd->setText("No Image loaded.");
		return osgDB::ReaderWriter::ReadResult::FILE_NOT_HANDLED;
	}
	//read image file

	std::ifstream filePath(file.c_str(), std::ios::in | std::ios::binary);

	if (!filePath) {
		osd->setText("No Image loaded.");
		return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}

	// try jp2 (jasper) since I heard it is better
	osgDB::ReaderWriter* rw;
	rw = osgDB::Registry::instance()->getReaderWriterForExtension("jp2");
	if (!rw) {
		rw = osgDB::Registry::instance()->getReaderWriterForExtension("jpg");
		std::cout << "No jp2 plugin found! Falling back to jp2 plugin."
				<< std::endl;
	}
	// fallback to jpeg
	if (!rw) {
		osd->setText("No jpg plugin found!");
		std::cout << "No jpg plugin found!!!" << std::endl;
		return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}
	std::cout << rw->className() << std::endl;

	osgDB::ReaderWriter::ReadResult rs = rw->readImage(filePath, 0);

	//check image
	if (!rs.validImage()) {
		osd->setText("Error loading Image!");
		std::cout << "Error loading Image: " << file.c_str() << "!"
				<< std::endl;
		std::cout << rs.error() << std::endl;
		//return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}

	// split image
	osg::Image* imagel = rs.takeImage();
	//image->scaleImage(image->s()/8*8,image->t()/4*4,image->r());
	int internalFormat = imagel->getInternalTextureFormat();
	if (internalFormat > 4)
		internalFormat = 3; // is internalFormat broken in ReaderWriterJPEG?
	int s = imagel->s();
	int t = imagel->t();

	filePath.clear();
	// move to 2nd image (starts with FFD8FFE1 )
	// filePath.seekg(0,std::ifstream::beg);
	int fileSize = filePath.tellg();
	//filePath.seekg(4888064,std::ifstream::beg);
	// seek(&filePath,0xFFD8FFE1,2); //2nd startjpeg
	char pattern[] = { 0xFF, 0xD8, 0xFF, 0xE1 };//{ 'F', 'F', 'D', '8', 'F', 'F', 'E', '1' };//{0xFF,0xD8,0xFF,0xE1};
	int pos = findPattern(&filePath, pattern, 4* fileSize / 10, 4); // 4/5 of file should be enough to find start of 2nd image
	filePath.clear();
	filePath.seekg(pos, std::ifstream::beg); //2nd startjpeg

	rs = rw->readImage(filePath, 0);

	//check image
	if (!rs.validImage()) {
		osd->setText("Error loading Image!");
		std::cout << "Error loading Image: " << file.c_str() << "!"
				<< std::endl;
		std::cout << rs.error() << std::endl;
		//return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
	}

	osg::Image* imager = rs.takeImage();

	unsigned char* data;
	unsigned char* datal = new unsigned char[internalFormat * s * t];
	unsigned char* datar = new unsigned char[internalFormat * s * t];

	for (int rows = 0; rows < t; rows++) {
		data = imagel->data();
		memcpy(datal + internalFormat * rows * s, data + internalFormat * (rows
				* s), internalFormat * s);
		data = imager->data();
		memcpy(datar + internalFormat * rows * s, data + internalFormat * (rows
				* s), internalFormat * s);
	}

	unsigned int pixelFormat = internalFormat == 1 ? GL_LUMINANCE
			: internalFormat == 2 ? GL_LUMINANCE_ALPHA
					: internalFormat == 3 ? GL_RGB
							: internalFormat == 4 ? GL_RGBA : (GLenum) -1;

	unsigned int dataType = GL_UNSIGNED_BYTE;

	//apply
	textureLeft->setTextureSize(s, t);
	textureRight->setTextureSize(s, t);

	left->setImage(s, t, 1, internalFormat, pixelFormat, dataType, datal,
			osg::Image::USE_NEW_DELETE);

	right->setImage(s, t, 1, internalFormat, pixelFormat, dataType, datar,
			osg::Image::USE_NEW_DELETE);

	textureLeft->dirtyTextureObject();
	textureRight->dirtyTextureObject();
	filePath.close();
	if (imagel->data())
		delete imagel->data();
	if (imager->data())
		delete imager->data();
	return osgDB::ReaderWriter::ReadResult::FILE_LOADED;
}

// read image and split into two
osgDB::ReaderWriter::ReadResult readImages(const std::string& file,
		osg::Image* left, osg::Image* right) {
	// manually reading the jps file since
	// osgDB::Registry::instance()->addFileExtensionAlias("jps", "jp2"); in connection with
	// osgDB::image=osgDB::readImageFile(file); does not work!!!

	// valid file?
	std::string ext = osgDB::getFileExtension(file);
	if (ext.find("jpeg") != ext.npos || ext.find("jpg") != ext.npos
			|| ext.find("JPG") != ext.npos || ext.find("jps") != ext.npos
			|| ext.find("JPS") != ext.npos) {
		readJPGImages(file, left, right);
	} else if (ext.find("mpo") != ext.npos || ext.find("MPO") != ext.npos) {
		readMPOImages(file, left, right);
	} else
		osd->setText("No Image loaded.");
	return osgDB::ReaderWriter::ReadResult::FILE_NOT_HANDLED;
	//	optimizer.optimize(root,osgUtil::Optimizer::ALL_OPTIMIZATIONS);
}

osg::Geode* createSectorForImage(osg::Texture2D* texture, osg::Image* image,
		float s, float t, float _radius, float _height, float _length,
		float parallax = 0.0f) {

	int segments = 20;
	float Theta = _length / _radius;
	float dTheta = Theta / (float) (segments - 1);

	float ThetaZero = _height * s / (t * _radius);

	// set up the texture.
	image->setDataVariance(osg::Object::DYNAMIC);
	texture->setFilter(osg::Texture2D::MIN_FILTER, osg::Texture2D::LINEAR);
	texture->setFilter(osg::Texture2D::MAG_FILTER, osg::Texture2D::LINEAR);

	texture->setImage(image);
	texture->setWrap(osg::Texture::WRAP_S, osg::Texture::CLAMP_TO_EDGE);
	texture->setResizeNonPowerOfTwoHint(false);
	texture->setFilter(osg::Texture2D::MIN_FILTER, osg::Texture2D::NEAREST);//
	texture->setFilter(osg::Texture2D::MAG_FILTER, osg::Texture2D::NEAREST);//
	texture->setBorderColor(osg::Vec4(0.0f, 0.0f, 0.0f, 1.0f));
	texture->setWrap(osg::Texture2D::WRAP_S, osg::Texture::CLAMP_TO_BORDER);
	texture->setWrap(osg::Texture2D::WRAP_T, osg::Texture::CLAMP_TO_BORDER);

	osg::StateSet* dstate = new osg::StateSet;
	dstate->setMode(GL_CULL_FACE, osg::StateAttribute::OFF);
	dstate->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
	dstate->setTextureAttributeAndModes(0, texture, osg::StateAttribute::ON);
	dstate->setDataVariance(osg::Object::DYNAMIC);

	osg::Geometry* geom = new osg::Geometry;
	geom->setStateSet(dstate);

	osg::Vec3Array* coords = new osg::Vec3Array();
	osg::Vec2Array* tcoords = new osg::Vec2Array();

	int i;
	float angle = -Theta / 2.0f;
	for (i = 0; i < segments; ++i, angle += dTheta) {
		coords->push_back(osg::Vec3(sinf(angle) * _radius, cosf(angle)
				* _radius, _height * 0.5f)); // top
		coords->push_back(osg::Vec3(sinf(angle) * _radius, cosf(angle)
				* _radius, -_height * 0.5f)); // bottom.

		tcoords->push_back(osg::Vec2(angle / ThetaZero + 0.5f + parallax, 1.0f)); // top
		tcoords->push_back(osg::Vec2(angle / ThetaZero + 0.5f + parallax, 0.0f)); // bottom.

	}

	osg::Vec4Array* colors = new osg::Vec4Array();
	colors->push_back(osg::Vec4(1.0f, 1.0f, 1.0f, 1.0f));

	osg::DrawArrays* elements = new osg::DrawArrays(
			osg::PrimitiveSet::QUAD_STRIP, 0, coords->size());

	geom->setVertexArray(coords);
	geom->setTexCoordArray(0, tcoords);
	geom->setColorArray(colors);
	geom->setColorBinding(osg::Geometry::BIND_OVERALL);

	geom->addPrimitiveSet(elements);

	// set up the geode.
	osg::Geode* geode = new osg::Geode;
	geode->addDrawable(geom);

	return geode;

}

// create a switch containing a set of child each containing a
// stereo image pair.
osg::Switch* createScene(float _radius, float _height, float _length,
		float parallax = 0.0f) {

	osg::Switch* sw = new osg::Switch;

	if (imageLeft.valid() && imageRight.valid()) {
		float average_s = (imageLeft->s() + imageRight->s()) * 0.5f;
		float average_t = (imageLeft->t() + imageRight->t()) * 0.5f;
		//((osg::Texture2D*)textureLeft.get())->setResizeNonPowerOfTwoHint(0);
		//((osg::Texture2D*)textureRight.get())->setResizeNonPowerOfTwoHint(0);
		//((osg::Texture2D*)textureLeft.get())->setInternalFormat(GL_COMPRESSED_RGB_ARB);
		//((osg::Texture2D*)textureRight.get())->setInternalFormat(GL_COMPRESSED_RGB_ARB);
		osg::Geode* geodeLeft = createSectorForImage(
				(osg::Texture2D*) textureLeft.get(), imageLeft.get(),
				average_s, average_t, _radius, _height, _length, -parallax);
		geodeLeft->setNodeMask(0x01);

		osg::Geode* geodeRight = createSectorForImage(
				(osg::Texture2D*) textureRight.get(), imageRight.get(),
				average_s, average_t, _radius, _height, _length, parallax);
		geodeRight->setNodeMask(0x02);

		osg::ref_ptr<osg::Group> imageGroup = new osg::Group;

		imageGroup->addChild(geodeLeft);
		imageGroup->addChild(geodeRight);

		sw->addChild(imageGroup.get());
	} else {
		std::cout << "Warning: Unable to load image files, '" << fileList[0]
				<< ", required for stereo imaging." << std::endl;
	}

	if (sw->getNumChildren() > 0) {
		// select first child.
		sw->setSingleChildOn(0);
	}

	return sw;
}

void loadImage() {
	readImages(fileList[filenum], (osg::Image*) imageLeft.get(),
			(osg::Image*) imageRight.get());
	rootNode = createScene(_radius, _height, _length, parallax);
	root->removeChild(1);
	root->addChild(rootNode.get());
	((osg::Switch *) root)->setAllChildrenOn();
	sprintf(msgString, "%d/%d %s", filenum + 1, fileList.size(),
			fileList[filenum].c_str());
	std::string msg = "";
	msg.append(msgString);
	osd->setText(msg, 7);
	newImage = false;
}

void file_up(int count) {
	if (filenum < fileList.size() - 1 || filenum == -1) {
		//((osg::Switch *) root)->setAllChildrenOff();
		filenum = filenum + count;
		if (filenum > fileList.size() - 1)
			filenum = fileList.size() - 1;
		osd->setText("Loading...");
		newImage = true;
	} else {
		if (slideshow) {
			filenum = -1;
			return;
		}
		osd->setText("No more files.", 3);

	}
}

void file_down(int count) {
	if (filenum > 0) {
		//((osg::Switch *) root)->setAllChildrenOff();
		filenum = filenum - count;
		if (filenum < 0 || filenum > fileList.size() - 1)
			filenum = 0;
		osd->setText("Loading...");
		newImage = true;

	} else {
		osd->setText("Already at first file.", 3);
	}
}

//update filter uniforms
void updateFilter(osg::StateSet* ss) {
	if (!shader)
		return;
	switch (filtermode) {
	case NONE:
		ss->addUniform(new osg::Uniform("red", osg::Vec3(1.0f, 0.0f, 0.0f)));
		ss->addUniform(new osg::Uniform("green", osg::Vec3(0.0f, 1.0f, 0.0f)));
		ss->addUniform(new osg::Uniform("blue", osg::Vec3(0.0f, 0.0f, 1.0f)));
		osd->setText("No Filter", 3);
		break;

	case BW:
		ss->addUniform(new osg::Uniform("red", osg::Vec3(0.3f, 0.59f, 0.11f)));
		ss->addUniform(new osg::Uniform("green", osg::Vec3(0.3f, 0.59f, 0.11f)));
		ss->addUniform(new osg::Uniform("blue", osg::Vec3(0.3f, 0.59f, 0.11f)));
		osd->setText("Black/White Filter", 3);
		break;

	case SEPIA:
		ss->addUniform(new osg::Uniform("red", osg::Vec3(0.3f, 0.59f, 0.11f)
				* 1.0f));
		ss->addUniform(new osg::Uniform("green", osg::Vec3(0.3f, 0.59f, 0.11f)
				* 0.83f));
		ss->addUniform(new osg::Uniform("blue", osg::Vec3(0.3f, 0.59f, 0.11f)
				* 0.71f));
		osd->setText("Sepia Filter", 3);
		break;

	default:
		break;
	}
	ss->addUniform(new osg::Uniform("gamma", gammaval));
	ss->addUniform(new osg::Uniform("brightness", brightness));
	ss->addUniform(new osg::Uniform("contrast", contrast));
}

// initialize filter shader
int initFilterShader(osg::StateSet* ss) {
	std::cout << "Initializing shaders..." << std::endl;
	pgm = new osg::Program;
	pgm->setName("Filter");
	pgm->addShader(new osg::Shader(osg::Shader::VERTEX, vsrc));
	pgm->addShader(new osg::Shader(osg::Shader::FRAGMENT, fsrc));
	ss->setAttributeAndModes(pgm, osg::StateAttribute::ON);
	updateFilter(ss);

	return 0;
}

// input handling
bool handlescroll = true;
bool myKeyboardEventHandler::handle(const osgGA::GUIEventAdapter& ea,
		osgGA::GUIActionAdapter& aa) {
	switch (ea.getEventType()) {

	case osgGA::GUIEventAdapter::SCROLL:
		if (!handlescroll) { // scrolling events occur double, why?
			handlescroll = true;
			return true;
		}
		handlescroll = false;
		switch (ea.getScrollingMotion()) {
		case osgGA::GUIEventAdapter::SCROLL_DOWN:

			if (!newImage)
				file_down(1); // the if is neccesary to protect from double wheel events
			return true;
			break;

		case osgGA::GUIEventAdapter::SCROLL_UP:
			if (!newImage)
				file_up(1);
			return true;
			break;

		default:
			break;
		}
		break;

	case (osgGA::GUIEventAdapter::KEYDOWN): {
		std::string msg = "";
		switch (ea.getKey()) {
		case osgGA::GUIEventAdapter::KEY_Home:
			if (!newImage)
				file_down(INT_MAX);
			break;
		case osgGA::GUIEventAdapter::KEY_End:
			if (!newImage)
				file_up(INT_MAX);
			break;
		case osgGA::GUIEventAdapter::KEY_Down:
			if (!newImage && !(ea.getModKeyMask()
					& osgGA::GUIEventAdapter::MODKEY_SHIFT))
				file_down(1);
			else
				file_down(10);
			break;

		case osgGA::GUIEventAdapter::KEY_Up:
			if (!newImage && !(ea.getModKeyMask()
					& osgGA::GUIEventAdapter::MODKEY_SHIFT))
				file_up(1);
			else
				file_up(10);
			break;

		case 'w':
			return true;
		case 'l':
			return true;
		case 't':
			filtermode++;
			filtermode = filtermode % 3;
			updateFilter(root->getOrCreateStateSet());
			return true;
		case osgGA::GUIEventAdapter::KEY_KP_Enter:
		case osgGA::GUIEventAdapter::KEY_Return:
			slideshow = slideshow == true ? false : true;
			if (slideshow)
				osd->setText("Slideshow enabled", 5);
			else
				osd->setText("Slideshow disabled", 5);
			break;
		case '+':
		case osgGA::GUIEventAdapter::KEY_KP_Add:
			parallax = parallax + 0.001f;
			rootNode = createScene(_radius, _height, _length, parallax);
			root->removeChild(1);
			root->addChild(rootNode.get());
			((osg::Switch *) root)->setAllChildrenOn();
			sprintf(msgString, "Parallax: %+1.3f", parallax);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		case '-':
		case osgGA::GUIEventAdapter::KEY_KP_Subtract:
			parallax = parallax - 0.001f;
			rootNode = createScene(_radius, _height, _length, parallax);
			root->removeChild(1);
			root->addChild(rootNode.get());
			((osg::Switch *) root)->setAllChildrenOn();
			sprintf(msgString, "Parallax: %+1.3f", parallax);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;
		case osgGA::GUIEventAdapter::KEY_F1:
			brightness -= 0.005;
			if (brightness < -1)
				brightness = -1;
			updateFilter(root->getOrCreateStateSet());
			sprintf(msgString, "Brightness: %+1.3f", brightness);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;
		case osgGA::GUIEventAdapter::KEY_F2:
			brightness += 0.005;
			if (brightness > 1)
				brightness = 1;
			updateFilter(root->getOrCreateStateSet());
			sprintf(msgString, "Brightness: %+1.3f", brightness);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;
		case osgGA::GUIEventAdapter::KEY_F3:
			contrast -= 0.005;
			if (contrast < 0)
				contrast = 0;
			updateFilter(root->getOrCreateStateSet());
			sprintf(msgString, "Contrast: %1.3f", contrast);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		case osgGA::GUIEventAdapter::KEY_F4:
			contrast += 0.005;
			if (contrast > 2)
				contrast = 2;
			updateFilter(root->getOrCreateStateSet());
			sprintf(msgString, "Contrast: %1.3f", contrast);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		case osgGA::GUIEventAdapter::KEY_Space:
			filtermode = NONE;
			gammaval = 1.0f;
			contrast = 1.0f;
			brightness = 0.0f;
			parallax = 0.0f;
			updateFilter(root->getOrCreateStateSet());
			rootNode = createScene(_radius, _height, _length, parallax);
			root->removeChild(1);
			root->addChild(rootNode.get());
			((osg::Switch *) root)->setAllChildrenOn();
			//return false;
			break;

		case osgGA::GUIEventAdapter::KEY_F5:
			gammaval -= 0.01;
			if (gammaval < -1)
				gammaval = -1;
			updateFilter(root->getOrCreateStateSet());
			sprintf(msgString, "Gamma: %1.3f", gammaval);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		case osgGA::GUIEventAdapter::KEY_F6:
			gammaval += 0.01;
			if (gammaval > 5)
				gammaval = 5;
			updateFilter(root->getOrCreateStateSet());
			sprintf(msgString, "Gamma: %1.3f", gammaval);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		case osgGA::GUIEventAdapter::KEY_F7:
			slideDelay -= 1;
			if (slideDelay < 0)
				slideDelay = 0;
			sprintf(msgString, "Slideshow delay: %1.0fs", slideDelay);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		case osgGA::GUIEventAdapter::KEY_F8:
			slideDelay += 1;
			if (slideDelay > 3600)
				slideDelay = 3600;
			sprintf(msgString, "Slideshow delay: %1.0fs", slideDelay);
			msg.append(msgString);
			osd->setText(msg, 3);
			break;

		default:
			return false;
		}
		std::cout << "parallax: " << parallax << "  gamma: " << gammaval
				<< "  brightness: " << brightness << "  contrast: " << contrast
				<< "  filter: " << filtermode << std::endl;

	}
	default:
		return false;
	}
	return true;
}

int main(int argc, char **argv) {
	// set default stereo mode

	osg::DisplaySettings::instance()->setStereo(true);
	osg::DisplaySettings::instance()->setStereoMode(
			osg::DisplaySettings::QUAD_BUFFER);
	osg::DisplaySettings::instance()->setEyeSeparation(0.0f);

	// use an ArgumentParser object to manage the program arguments.
	osg::ArgumentParser arguments(&argc, argv);

	// construct the viewer.
	osgViewer::Viewer viewer(arguments);

	// set up the usage document, in case we need to print out how to use this program.
	arguments.getApplicationUsage()->setDescription(
			arguments.getApplicationName()
					+ "- Stereoscopic Image Viewer\n(C) 2009 Juergen Loeb");
	arguments.getApplicationUsage()->setCommandLineUsage(
			arguments.getApplicationName()
					+ " [options] stereofile(s)\n\nSIV -  Stereoscopic Image Viewer\n(C) 2009-2010 Jürgen Löb\njloeb(at)main-host.de\n");
	arguments.getApplicationUsage()->addCommandLineOption("-s",
			"start in slideshow mode");
	arguments.getApplicationUsage()->addCommandLineOption("-t",
			"delay for slideshow mode, defaults to 15 sec");
	arguments.getApplicationUsage()->addCommandLineOption(
			"--stereo",
			"Use default stereo mode which is\nQUAD_BUFFER if not overriden by\nenvironmental variable");
	arguments.getApplicationUsage()->addCommandLineOption("-d",
			"distance (values between 0.0 and 1.0 are senseful) ");
	arguments.getApplicationUsage()->addCommandLineOption("--vr920",
			"enable headtracking support for the vuzix vr920 hmd");
	arguments.getApplicationUsage()->addCommandLineOption("--disable-shaders",
			"disable shaders for unsupported graphics boards");
	arguments.getApplicationUsage()->addCommandLineOption(
			"--filter=[bw|sepia]", "enable filter on startup");
	arguments.getApplicationUsage()->addCommandLineOption("-m",
			"non default multicast ip");
	arguments.getApplicationUsage()->addCommandLineOption("-p",
			"non default multicast port");
	arguments.getApplicationUsage()->addCommandLineOption("-h or --help",
			"Display this information");

	arguments.getApplicationUsage()->addKeyboardMouseBinding("+/-",
			"increase/decrease parallax");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("HOME/END",
			"display first/last image");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("UP/DOWN",
			"display next/previous image");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("SHIFT+UP/DOWN",
			"jump 10 images forward/backward");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("F1/F2",
			"decrease/increase brightness");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("F3/F4",
			"decrease/increase contrast");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("F5/F6",
			"decrease/increase gamma");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("F5/F6",
			"decrease/increase slideshow delay");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("t",
			"toggle filter (normal/bw/sepia)");
	arguments.getApplicationUsage()->addKeyboardMouseBinding("[Enter]",
			"toggle slideshow on/off");

	while (arguments.read("-t", slideDelay)) {
	}

	// slidshow?
	while (arguments.read("-s"))
		slideshow = true;

	//vr920 support
	while (arguments.read("--vr920"))
		vr920 = true;

	//shader?
	while (arguments.read("--disable-shaders"))
		shader = false;

	//bw?
	while (arguments.read("--filter=bw"))
		filtermode = BW;

	//sepia?
	while (arguments.read("--filter=sepia"))
		filtermode = SEPIA;

	//non default multicast ip?
	while (arguments.read("-m", multicastAddress)) {
	}

	//non default multicats port?
	while (arguments.read("-p", port)) {
	}

	//non default distance?
	while (arguments.read("-d", distance)) {
	}

	keyswitchManipulator = new osgGA::KeySwitchMatrixManipulator;
	char manipulator = '1';
	if (vr920) {
		keyswitchManipulator->addMatrixManipulator(manipulator, "VR920",
				new VR920TrackingManipulator((char *) multicastAddress.c_str(),
						port));
		manipulator++;
	}

	keyswitchManipulator->addMatrixManipulator(manipulator, "Zoom",
			new ZoomManipulator());
	manipulator++;

	keyswitchManipulator->addMatrixManipulator(manipulator, "Trackball",
			new osgGA::TrackballManipulator());

	std::string pathfile;
	char keyForAnimationPath = '5';
	while (arguments.read("-p", pathfile)) {
		osgGA::AnimationPathManipulator* apm =
				new osgGA::AnimationPathManipulator(pathfile);
		if (apm || !apm->valid()) {
			unsigned int num = keyswitchManipulator->getNumMatrixManipulators();
			keyswitchManipulator->addMatrixManipulator(keyForAnimationPath,
					"Path", apm);
			keyswitchManipulator->selectMatrixManipulator(num);
			++keyForAnimationPath;
		}
	}

	viewer.setCameraManipulator(keyswitchManipulator.get());
	keyswitchManipulator->setHomePosition(osg::Vec3(0.0f, 1 - distance * 2,
			0.0f), osg::Vec3(0.0f, 10.0f, 0.0f), osg::Vec3(0.0f, 0.0f, 1.0f),
			false);

	// add the thread model handler
	viewer.addEventHandler(new osgViewer::ThreadingHandler);

	// add the stats handler
	viewer.addEventHandler(new osgViewer::StatsHandler);

	// add the window size toggle handler
	viewer.addEventHandler(new osgViewer::WindowSizeHandler);

	// add the help handler
	viewer.addEventHandler(new osgViewer::HelpHandler(
			arguments.getApplicationUsage()));
	//viewer.getCameraManipulator()->setHomePosition(osg::Vec3f(0,-distance,,0),osg::Vec3f(0,0,0),osg::Vec3f(0,1,0),false);
	// keyboard handling
	myKeyboardEventHandler* keh = new myKeyboardEventHandler();

	viewer.getEventHandlers().push_front(keh);
	viewer.getDisplaySettings()->setThreadSafeReferenceCounting(true);
	osgViewer::GraphicsWindow
			* win =
					dynamic_cast<osgViewer::GraphicsWindow*> (viewer.getCamera()->getGraphicsContext());
	//if (win) win->setWindowName("new Name");
	if (win)
		win->setWindowRectangle(10, 10, 810, 610);
	//viewer.getCamera()->getGraphicsContext()->getTraits()->vsync
	//viewer.setUpViewOnSingleScreen(0);

	// get details on keyboard and mouse bindings used by the viewer.
	viewer.getUsage(*arguments.getApplicationUsage());

	// if user request help write it out to cout.
	if (arguments.read("-h") || arguments.read("--help")) {
		arguments.getApplicationUsage()->write(std::cout);
		return 1;
	}

	// any option left unread are converted into errors to write out later.
	arguments.reportRemainingOptionsAsUnrecognized();

	// report any errors if they have occured when parsing the program aguments.
	if (arguments.errors()) {
		arguments.writeErrorMessages(std::cout);
		return 1;
	}

	if (arguments.argc() <= 1) {
		arguments.getApplicationUsage()->write(std::cout,
				osg::ApplicationUsage::COMMAND_LINE_OPTION);
		return 1;
	}

	// extract the filenames from the arguments list.

	for (int pos = 1; pos < arguments.argc(); ++pos) {
		if (arguments.isString(pos))
			fileList.push_back(arguments[pos]);
	}

	if (fileList.size() < 1) {
		return 1;
	}

	// set up the use of stereo by default.
	osg::DisplaySettings* ds = viewer.getDisplaySettings();
	if (!ds)
		ds = osg::DisplaySettings::instance();
	if (ds)
		ds->setStereo(true);

	viewer.setThreadingModel(osgViewer::ViewerBase::SingleThreaded);
	// realize the viewer
	viewer.realize();
	if (!viewer.isRealized()) {
		ds->setStereoMode(osg::DisplaySettings::ANAGLYPHIC);
		std::cout
				<< "Problem starting the viewer in requested stereo mode. Falling back to  ANAGLYPHIC.\nCheck parameters and/or OSG_STEREO_MODE environment variable."
				<< std::endl;
		viewer.realize();
	}

	// init display
	osd = new OSD;
	osd->setColor(osg::Vec4d(0.80, 0.85, 1.0, 0.6));
	osd->setTextSize(28);
	// no shader use for osd
	osg::StateSet* osdstates = osd->getOrCreateStateSet();
	if (shader)
		osdstates->setAttributeAndModes(new osg::Program());

	osdstates->setMode(GL_DEPTH_TEST, osg::StateAttribute::OFF);
	osdstates->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);
	osdstates->setRenderBinDetails(11, "RenderBin");
	osdstates->setMode(GL_BLEND, osg::StateAttribute::ON);
	// create the scene.
	readImages(fileList[filenum], (osg::Image*) imageLeft.get(),
			(osg::Image*) imageRight.get());
	rootNode = createScene(_radius, _height, _length);

	// set the scene to render
	root->addChild(osd.get());
	root->addChild(rootNode.get());
	root->setDataVariance(osg::Object::DYNAMIC);
	osd->setPosition(osg::Vec3d(10.0f, 10.0f, 0.0f));

	//filter shader
	if (shader)
		initFilterShader(root->getOrCreateStateSet());

	viewer.setSceneData(root);
	viewer.getCamera()->setClearColor(osg::Vec4(0., 0., 0., 1.));
	viewer.getCamera()->setCullMask(0xffffffff);
	viewer.getCamera()->setCullMaskLeft(0x00000001);
	viewer.getCamera()->setCullMaskRight(0x00000002);

	osg::Timer timer;
	osg::Timer_t start_tick = timer.tick();
	osd->setText("SIV: press UP/DOWN to select Image");
	while (!viewer.done()) {
		//viewer.updateTraversal();
		if (newImage)
			loadImage();
		if (slideshow) {
			osg::Timer_t end_tick = timer.tick();
			if (timer.delta_s(start_tick, end_tick) > slideDelay) {
				file_up(1);
				start_tick = end_tick;
			}
		}
		viewer.frame();
	}

	return 0;
}
