/*
 # (C) 2009-2010 Jürgen Löb
 #
 # This program is distributed in the hope that it will be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 # See the Attribution-NonCommercial-ShareAlike 3.0 Unported for more details.
 #
 */
#include "ZoomManipulator.hpp"
#include "osd.hpp"
#include <iostream>
#include <stdlib.h>
#include <stdio.h>

extern osg::ref_ptr<OSD> osd;

ZoomManipulator::ZoomManipulator() :
	osgGA::TrackballManipulator() {

}

ZoomManipulator::~ZoomManipulator() {

}

bool ZoomManipulator::handle(const osgGA::GUIEventAdapter& ea,
		osgGA::GUIActionAdapter& us) {

	switch (ea.getEventType()) {
	case (GUIEventAdapter::FRAME):
		if (_thrown) {
			if (calcMovement())
				us.requestRedraw();
		}
		return false;
	default:
		break;
	}

	switch (ea.getEventType()) {

	case (GUIEventAdapter::DRAG): {
		addMouseEvent(ea);
		if (calcMovement())
			us.requestRedraw();
		us.requestContinuousUpdate(false);
		_thrown = false;
		return true;
	}

	case (GUIEventAdapter::KEYDOWN):
		if (ea.getKey() == GUIEventAdapter::KEY_Space) {
			flushMouseEventStack();
			_thrown = false;
			home(ea, us);
			return true;
		}
		return false;

	default:
		return false;
	}

	if (ea.getHandled())
		return false;

}

bool ZoomManipulator::calcMovement() {
	// return if less then two events have been added.
	if (_ga_t0.get() == NULL || _ga_t1.get() == NULL)
		return false;

	float dx = _ga_t0->getXnormalized() - _ga_t1->getXnormalized();
	float dy = _ga_t0->getYnormalized() - _ga_t1->getYnormalized();

	float distance = sqrtf(dx * dx + dy * dy);
	// return if movement is too fast, indicating an error in event values or change in screen.
	if (distance > 0.5) {
		return false;
	}

	// return if there is no movement.
	if (distance == 0.0f) {
		return false;
	}

	unsigned int buttonMask = _ga_t1->getButtonMask();
	if (buttonMask == GUIEventAdapter::MIDDLE_MOUSE_BUTTON || buttonMask
			== (GUIEventAdapter::LEFT_MOUSE_BUTTON
					| GUIEventAdapter::RIGHT_MOUSE_BUTTON)) {

		// pan model.

		float scale = -0.1f * _distance;

		//osg::Matrix rotation_matrix;
		//rotation_matrix.makeRotate(_rotation);

		osg::Vec3 dv(dx * scale, dy * scale, 0.0f);

		//_center += dv * rotation_matrix;
		_center += dv;
		return true;

	} else if (buttonMask == GUIEventAdapter::RIGHT_MOUSE_BUTTON) {

		// zoom model.

		float scale = 1.0f + dy * 0.3f;

		// notify(DEBUG_INFO) << "Pushing forward"<<std::endl;
		// push the camera forward.
		//float scale = -fd;

		//osg::Matrix rotation_matrix(_rotation);

		//osg::Vec3 dv = (osg::Vec3(0.0f, -1.0f, 0.0f) * rotation_matrix) * (dy* scale);
		osg::Vec3 dv = (osg::Vec3(0.0f, 0.0f, 1.0f)) * (dy * scale);

		_center += dv;

		return true;

	}

	return false;
}

void ZoomManipulator::setByMatrix(const osg::Matrixd& matrix) {
	//   _eye = matrix.getTrans();
	//   _rotation = matrix.getRotate();
	//   _distance = 1.0f;
	//   _baserot = osg::Matrixd::identity().rotate(_rotation);
}

void ZoomManipulator::setByInverseMatrix(const osg::Matrixd& matrix) {
	//   _eye = matrix.getTrans();
	//   _rotation = matrix.getRotate();
	//   _distance = 1.0f;
	//   _baserot = osg::Matrixd::identity().rotate(_rotation);
}

osg::Matrixd ZoomManipulator::getMatrix() const {
	// return osg::Matrixd::rotate(_pitch,1.0,0.0,0.0)*osg::Matrixd::rotate(_rotation)*osg::Matrixd::translate(_eye);
	return osg::Matrixd::translate(-_center) * osg::Matrixd::rotate(_rotation)
			* osg::Matrixd::translate(_eye);
}

osg::Matrixd ZoomManipulator::getInverseMatrix() const {
	// return osg::Matrixd::rotate(_pitch,1.0,0.0,0.0)*osg::Matrixd::rotate(_rotation)*osg::Matrixd::translate(_eye);
	return osg::Matrixd::inverse(osg::Matrixd::translate(-_center)
			* osg::Matrixd::rotate(_rotation) * osg::Matrixd::translate(_eye));
}

void ZoomManipulator::init(const GUIEventAdapter&, GUIActionAdapter&) {

}

void ZoomManipulator::home(double /*currentTime*/) {
	if (getAutoComputeHomePosition())
		computeHomePosition();
	computePosition(_homeEye, _homeCenter, _homeUp);
	_center = osg::Vec3f(0, 0, 0);
	//_center=_homeCenter;
	// _thrown = false;
}

void ZoomManipulator::home(const GUIEventAdapter& ea, GUIActionAdapter& us) {
	home(ea.getTime());

	us.requestRedraw();
	us.requestContinuousUpdate(true);
	us.requestWarpPointer((ea.getXmin() + ea.getXmax()) / 2.0f, (ea.getYmin()
			+ ea.getYmax()) / 2.0f);
}

void ZoomManipulator::computePosition(const osg::Vec3& eye,
		const osg::Vec3& center, const osg::Vec3& up) {

	osg::Vec3d lv = center - eye;

	osg::Vec3 f(lv);
	f.normalize();
	osg::Vec3 s(f ^ up);
	s.normalize();
	osg::Vec3 u(s ^ f);
	u.normalize();

	osg::Matrixd rotation_matrix(s[0], u[0], -f[0], 0.0f, s[1], u[1], -f[1],
			0.0f, s[2], u[2], -f[2], 0.0f, 0.0f, 0.0f, 0.0f, 1.0f);

	_eye = eye;
	_distance = lv.length();
	_rotation = rotation_matrix.getRotate().inverse();
	_baserot = osg::Matrixd::identity().rotate(_rotation);
}
