/*
# (C) 2009 Jürgen Löb 
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
# See the Attribution-NonCommercial-ShareAlike 3.0 Unported for more details.
#
*/
#include <osg/Switch>
#include <osgText/Text>
#include <osgGA/GUIEventHandler>
 	
#include <osgViewer/Viewer>
#include <osgViewer/GraphicsWindow>
#include <osg/Geometry>
#include <osg/Image>
#include <osg/BufferObject>
#include <osgGA/TrackballManipulator>
#include <osgGA/FlightManipulator>
#include <osgGA/DriveManipulator>
#include <osgGA/KeySwitchMatrixManipulator>
#include <osgGA/StateSetManipulator>
#include <osgGA/AnimationPathManipulator>
#include <osgGA/TerrainManipulator>
#include <osg/TexMat>
#include <osgViewer/ViewerEventHandlers>
#include <osg/Texture2D>
#include <osgUtil/SceneView>
#include <osgUtil/Statistics>
#include <osgDB/ReadFile>
#include <iostream>
#include <stdio.h>
#include <string.h>

#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>
#include <osgGA/GUIEventAdapter>
#include <string>
#include <sstream>

#include "osd.hpp"

#include "VR920TrackingManipulator.hpp"

std::vector<std::string>::size_type  filenum=0;
typedef std::vector<std::string> FileList;
FileList fileList;
bool slideshow = false;
bool vr920=false;
float distance=1.0f;
float slideDelay = 10.0f;

float _fovy = 1.0f;
float _radius = 1.0f;
float _height = 2*_radius*tan(_fovy*0.5f);
float _length = osg::PI*_radius;  // half a cylinder.

osg::ref_ptr<osg::Image>imageLeft = new osg::Image();
osg::ref_ptr<osg::Image>imageRight  =  new osg::Image();
osg::ref_ptr<osg::Texture2D>textureLeft = new osg::Texture2D;
osg::ref_ptr<osg::Texture2D>textureRight = new osg::Texture2D;
osg::Switch* root=new osg::Switch();
OSD osd;
osg::ref_ptr<osg::Switch> rootNode;

std::string  multicastAddress="224.0.0.42";
int port=4242;

class myKeyboardEventHandler : public osgGA::GUIEventHandler
{
public:
   virtual bool handle(const osgGA::GUIEventAdapter& ea,osgGA::GUIActionAdapter&);
   virtual void accept(osgGA::GUIEventHandlerVisitor& v)   { v.visit(*this); };
};



// read image and split into two
osgDB::ReaderWriter::ReadResult  readImages(const std::string& file, osg::Image* left, osg::Image* right) 
         {	
     		// manually reading the jps file since 
     		// osgDB::Registry::instance()->addFileExtensionAlias("jps", "jp2"); in connection with
     		// osgDB::image=osgDB::readImageFile(file); does not work!!!
     		
     		// valid file?
     		std::string ext = osgDB::getFileExtension(file);
         	if (ext.find("jpeg")==ext.npos&&ext.find("jpg")==ext.npos&&ext.find("JPG")==ext.npos&&ext.find("jps")==ext.npos&&ext.find("JPS")==ext.npos){
         	   	 osd.setText("No Image loaded.");
         		 return osgDB::ReaderWriter::ReadResult::FILE_NOT_HANDLED;
         	}
         	
         	//read image file

         	std::ifstream filePath(file.c_str(), std::ios::in | std::ios::binary);
         	
         	if(!filePath) {
         		osd.setText("No Image loaded.");
         		return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
         	} 	
         	
         	// try jp2 (jasper) since I heard it is better
         	osgDB::ReaderWriter* rw;
            rw=osgDB::Registry::instance()->getReaderWriterForExtension("jp2");
            if(!rw) {
            	rw=osgDB::Registry::instance()->getReaderWriterForExtension("jpg");
            	std::cout<<"No jp2 plugin found! Falling back to jp2 plugin."<<std::endl;
            }
            // fallback to jpeg
			if(!rw){
			osd.setText("No jpg plugin found!");
			std::cout<<"No jpg plugin found!!!"<<std::endl;
			return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
			}
			std::cout << rw->className() << std::endl;
         	
         	osgDB::ReaderWriter::ReadResult rs=rw->readImage(filePath,0);
         	
         	//check image
         	if(!rs.validImage()){
         	osd.setText("Error loading Image!");
			std::cout<<"Error loading Image: "<< file.c_str() << "!"<<std::endl;
			std::cout << rs.error() <<std::endl;
			//return osgDB::ReaderWriter::ReadResult::ERROR_IN_READING_FILE;
         	}
         	
         	// split image
   			osg::Image* image = rs.takeImage();
   			//image->scaleImage(image->s()/8*8,image->t()/4*4,image->r());
   			int internalFormat = image->getInternalTextureFormat();
   			if (internalFormat>4)internalFormat=3; // is internalFormat broken in ReaderWriterJPEG?
            int s = image->s();
            int t = image->t();
            unsigned char* data;
            unsigned char* datal = new unsigned char[internalFormat*s*t/2];
            unsigned char* datar = new unsigned char[internalFormat*s*t/2];
 
            unsigned int pixelFormat =
                internalFormat == 1 ? GL_LUMINANCE :
                internalFormat == 2 ? GL_LUMINANCE_ALPHA :
                internalFormat == 3 ? GL_RGB :
                internalFormat == 4 ? GL_RGBA : (GLenum)-1;
 
            unsigned int dataType = GL_UNSIGNED_BYTE;
 			data=image->data();
 			for(int rows=0; rows < t; rows++){
 				memcpy(datal+internalFormat*rows*s/2,data+internalFormat*(rows*s),internalFormat*s/2);
 			 	memcpy(datar+internalFormat*rows*s/2,data+internalFormat*(rows*s+s/2),internalFormat*s/2);
 			}
 			 
 			//apply
 			textureLeft->setTextureSize(s/2,t);
 			textureRight->setTextureSize(s/2,t);
 			
            left->setImage(s/2,t,1,
                 internalFormat,
                 pixelFormat,
                 dataType,
                 datal,
                 osg::Image::USE_NEW_DELETE);

             right->setImage(s/2,t,1,
                 internalFormat,
                 pixelFormat,
                 dataType,
                 datar,
                 osg::Image::USE_NEW_DELETE);

 			if (data) delete data;
 			textureLeft->dirtyTextureObject();
 			textureRight->dirtyTextureObject();
            return osgDB::ReaderWriter::ReadResult::FILE_LOADED;
 
         }
         

osg::Geode* createSectorForImage(osg::Texture2D* texture, osg::Image* image,float s,float t, float _radius, float _height, float _length)
{
	
    int segments = 20;
    float Theta = _length/_radius;
    float dTheta = Theta/(float)(segments-1);
    
    float ThetaZero = _height*s/(t*_radius);
    
    // set up the texture.
    image->setDataVariance(osg::Object::DYNAMIC);
    texture->setFilter(osg::Texture2D::MIN_FILTER,osg::Texture2D::LINEAR);
    texture->setFilter(osg::Texture2D::MAG_FILTER,osg::Texture2D::LINEAR);
   

    texture->setImage(image);
    texture->setWrap(osg::Texture::WRAP_S, osg::Texture::CLAMP_TO_EDGE);
    texture->setResizeNonPowerOfTwoHint(false);
    texture->setFilter(osg::Texture2D::MIN_FILTER,osg::Texture2D::NEAREST);//
    texture->setFilter(osg::Texture2D::MAG_FILTER,osg::Texture2D::NEAREST);//
    texture->setBorderColor(osg::Vec4(0.0f,0.0f,0.0f,1.0f));
    texture->setWrap(osg::Texture2D::WRAP_S,osg::Texture::CLAMP_TO_BORDER);
    texture->setWrap(osg::Texture2D::WRAP_T,osg::Texture::CLAMP_TO_BORDER);

    osg::StateSet* dstate = new osg::StateSet;
    dstate->setMode(GL_CULL_FACE,osg::StateAttribute::OFF);
    dstate->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
    dstate->setTextureAttributeAndModes(0, texture,osg::StateAttribute::ON);
    dstate->setDataVariance(osg::Object::DYNAMIC); 


    osg::Geometry* geom = new osg::Geometry;
    geom->setStateSet(dstate);

    osg::Vec3Array* coords = new osg::Vec3Array();
    osg::Vec2Array* tcoords = new osg::Vec2Array();
   
    int i;
    float angle = -Theta/2.0f;
    for(i=0;
        i<segments;
        ++i, angle+=dTheta)
    {
        coords->push_back(osg::Vec3(sinf(angle)*_radius,cosf(angle)*_radius,_height*0.5f)); // top
        coords->push_back(osg::Vec3(sinf(angle)*_radius,cosf(angle)*_radius,-_height*0.5f)); // bottom.
        
        tcoords->push_back(osg::Vec2(angle/ThetaZero+0.5f,1.0f)); // top
        tcoords->push_back(osg::Vec2(angle/ThetaZero+0.5f,0.0f)); // bottom.

    }

    osg::Vec4Array* colors = new osg::Vec4Array();
    colors->push_back(osg::Vec4(1.0f,1.0f,1.0f,1.0f));

    osg::DrawArrays* elements = new osg::DrawArrays(osg::PrimitiveSet::QUAD_STRIP,0,coords->size());

    

    geom->setVertexArray(coords);
    geom->setTexCoordArray(0,tcoords);
    geom->setColorArray(colors);
    geom->setColorBinding(osg::Geometry::BIND_OVERALL);

    geom->addPrimitiveSet(elements);

    // set up the geode.
    osg::Geode* geode = new osg::Geode;
    geode->addDrawable(geom);

    return geode;

}

// create a switch containing a set of child each containing a 
// stereo image pair.
osg::Switch* createScene(const FileList& fileList, float _radius, float _height, float _length)
{
	
    osg::Switch* sw = new osg::Switch;
    
    // load the images.
        
    	readImages(fileList[filenum],(osg::Image*)imageLeft.get(),(osg::Image*)imageRight.get());
    	
        if (imageLeft.valid() && imageRight.valid())
        {
            float average_s = (imageLeft->s()+imageRight->s())*0.5f;
            float average_t = (imageLeft->t()+imageRight->t())*0.5f;
            //((osg::Texture2D*)textureLeft.get())->setResizeNonPowerOfTwoHint(0);
    	    //((osg::Texture2D*)textureRight.get())->setResizeNonPowerOfTwoHint(0);
    	    //((osg::Texture2D*)textureLeft.get())->setInternalFormat(GL_COMPRESSED_RGB_ARB);
    	    //((osg::Texture2D*)textureRight.get())->setInternalFormat(GL_COMPRESSED_RGB_ARB);
            osg::Geode* geodeLeft = createSectorForImage((osg::Texture2D*)textureLeft.get(), imageLeft.get(),average_s,average_t, _radius, _height, _length);
            geodeLeft->setNodeMask(0x01);

            osg::Geode* geodeRight = createSectorForImage((osg::Texture2D*)textureRight.get(), imageRight.get(),average_s,average_t, _radius, _height, _length);
            geodeRight->setNodeMask(0x02);

            osg::ref_ptr<osg::Group> imageGroup = new osg::Group;
            
            imageGroup->addChild(geodeLeft);
            imageGroup->addChild(geodeRight);
            
            sw->addChild(imageGroup.get());
        }
        else
        {
            std::cout << "Warning: Unable to load image files, '"<<fileList[0]<<", required for stereo imaging."<<std::endl;
        }
    

    if (sw->getNumChildren()>0)
    {
        // select first child.
        sw->setSingleChildOn(0);
    }

    return sw;
}

void file_up(){
	if(filenum<fileList.size()-1||filenum==-1){
            					osd.setText ("Loading...");
            					((osg::Switch *)root)->setAllChildrenOff();
                                filenum++;
                                char szString[200];
								sprintf( szString, "%d/%d %s", filenum+1,fileList.size(),fileList[filenum].c_str());
                                std::string msg ="";
                                msg.append(szString);
                                osd.setText (msg);
                                //readImages(fileList[filenum],0,(osg::Image*)imageLeft.get(),(osg::Image*)imageRight.get());
                                rootNode = createScene(fileList,_radius,_height,_length);
                                root->removeChild(1);
                                root->addChild(rootNode.get());
                                ((osg::Switch *)root)->setAllChildrenOn();
                                }
                         else{
                         		if(slideshow){
                         			filenum=-1;
                         			return;
                         		}
                                osd.setText ("No more files.");
                               
                         }
	}
	
void file_down(){
	 if(filenum>0){
                         		osd.setText ("Loading...");
                         		((osg::Switch *)root)->setAllChildrenOff();
                                filenum--;
                                char szString[200];
								sprintf( szString, "%d/%d %s", filenum+1,fileList.size(),fileList[filenum].c_str());
                                std::string msg ="";
                                msg.append(szString);
                                osd.setText (msg);
                                //readImages(fileList[filenum],0,(osg::Image*)imageLeft.get(),(osg::Image*)imageRight.get());
                                rootNode = createScene(fileList,_radius,_height,_length);
                                root->removeChild(1);
                                root->addChild(rootNode.get());
                                ((osg::Switch *)root)->setAllChildrenOn();
                                }
                         else{                         	
                                osd.setText ("Already at first file.");
                         }
}

// input handling
bool handlescroll=true;
bool myKeyboardEventHandler::handle(const osgGA::GUIEventAdapter& ea,osgGA::GUIActionAdapter& aa)
 {
   switch(ea.getEventType())
   {
   
         case osgGA::GUIEventAdapter::SCROLL:
         	if(!handlescroll){		// scrolling events occur double, why?
         		 handlescroll=true;
         		 return true;
         	}
         	handlescroll=false;
         	switch(ea.getScrollingMotion()) {
         case osgGA::GUIEventAdapter::SCROLL_DOWN:
		 	
		 	file_down();
		 	return true;
		 break;

         case osgGA::GUIEventAdapter::SCROLL_UP:
               file_up();
         return true;

      
		 break;
		 
		 default:
		 	break;
      }
      break;
         	
      case(osgGA::GUIEventAdapter::KEYDOWN):
      {	
         switch(ea.getKey()){   	
         	case osgGA::GUIEventAdapter::KEY_Down:
          	file_down();
		    break;     
                 
            case osgGA::GUIEventAdapter::KEY_Up:
            file_up();
		    break;
            
         default:
            return false;
         } 
      }
   default:
      return false;
   }
   return false;
}
 


int main( int argc, char **argv )
{	
	// set default stereo mode

	osg::DisplaySettings::instance()->setStereo(true); 
	osg::DisplaySettings::instance()->setStereoMode(osg::DisplaySettings::QUAD_BUFFER); 
	osg::DisplaySettings::instance()->setEyeSeparation(0.0f);
	
    // use an ArgumentParser object to manage the program arguments.
    osg::ArgumentParser arguments(&argc,argv);
    
    // set up the usage document, in case we need to print out how to use this program.
    arguments.getApplicationUsage()->setDescription(arguments.getApplicationName()+"- Stereoscopic Image Viewer\n(C) 2009 Jürgen Löb");
    arguments.getApplicationUsage()->setCommandLineUsage(arguments.getApplicationName()+" [options] jpsfile(s)\n\nSIV -  Stereoscopic Image Viewer\n(C) 2009 Jürgen Löb\njloeb(at)main-host.de\n");
    arguments.getApplicationUsage()->addCommandLineOption("-s","enable slideshow mode");
    arguments.getApplicationUsage()->addCommandLineOption("-t","delay for slideshow mode, defaults to 10 sec");
    arguments.getApplicationUsage()->addCommandLineOption("-d","distance (values between 0.0 and 1.0 are senseful) ");
    arguments.getApplicationUsage()->addCommandLineOption("--vr920","enable headtracking support for the vuzix vr920 hmd");
    arguments.getApplicationUsage()->addCommandLineOption("-m","non default multicast ip");
    arguments.getApplicationUsage()->addCommandLineOption("-p","non default multicast port");
    arguments.getApplicationUsage()->addCommandLineOption("-h or --help","Display this information");
    
    while (arguments.read("-t",slideDelay)) {}
    
	// slidshow?
    while (arguments.read("-s")) slideshow = true;
    
    //vr920 support
	while (arguments.read("--vr920")) vr920=true;
    
	//non default multicast ip?
	while (arguments.read("-m",multicastAddress)) {} 
	
	//non default multicats port?
	while (arguments.read("-p",port)) {}
	
	//non default distance?
	while (arguments.read("-d",distance)) {}
	
    // construct the viewer.
    osgViewer::Viewer viewer(arguments);
	
    osg::ref_ptr<osgGA::KeySwitchMatrixManipulator> keyswitchManipulator = new osgGA::KeySwitchMatrixManipulator;
    char manipulator='1';
    if (vr920)keyswitchManipulator->addMatrixManipulator( manipulator, "VR920",new VR920TrackingManipulator((char *)multicastAddress.c_str(),port));
    manipulator++;
    keyswitchManipulator->addMatrixManipulator( manipulator, "Trackball", new osgGA::TrackballManipulator() );
    manipulator++;
    keyswitchManipulator->addMatrixManipulator( manipulator, "Flight", new osgGA::FlightManipulator() );
    manipulator++;
    keyswitchManipulator->addMatrixManipulator( manipulator, "Drive", new osgGA::DriveManipulator() );
    manipulator++;
    keyswitchManipulator->addMatrixManipulator( manipulator, "Terrain", new osgGA::TerrainManipulator() );
	
    std::string pathfile;
    char keyForAnimationPath = '5';
    while (arguments.read("-p",pathfile))
    {
	    osgGA::AnimationPathManipulator* apm = new osgGA::AnimationPathManipulator(pathfile);
       if (apm || !apm->valid())
           {
	                unsigned int num = keyswitchManipulator->getNumMatrixManipulators();
 	                keyswitchManipulator->addMatrixManipulator( keyForAnimationPath, "Path", apm );
 	                keyswitchManipulator->selectMatrixManipulator(num);
 	                ++keyForAnimationPath;
 	            }
 	        }
 	
 	viewer.setCameraManipulator( keyswitchManipulator.get() );
 	keyswitchManipulator->setHomePosition(osg::Vec3(0.0f,1-distance*2,0.0f),osg::Vec3(0.0f,10.0f,0.0f),osg::Vec3(0.0f,0.0f,1.0f),false);
 	
 	// add the state manipulator
 	viewer.addEventHandler( new osgGA::StateSetManipulator(viewer.getCamera()->getOrCreateStateSet()) );
 	  
 	// add the thread model handler
 	viewer.addEventHandler(new osgViewer::ThreadingHandler);
 	
 	// add the stats handler
 	viewer.addEventHandler(new osgViewer::StatsHandler);
 	
 	// add the window size toggle handler
    viewer.addEventHandler(new osgViewer::WindowSizeHandler);
    
    // add the help handler
    viewer.addEventHandler(new osgViewer::HelpHandler(arguments.getApplicationUsage()));
 	//viewer.getCameraManipulator()->setHomePosition(osg::Vec3f(0,-distance,,0),osg::Vec3f(0,0,0),osg::Vec3f(0,1,0),false);
 	// keyboard handling
 	myKeyboardEventHandler* keh = new myKeyboardEventHandler();
 	viewer.getEventHandlers().push_back(keh);
 	viewer.getDisplaySettings()->setThreadSafeReferenceCounting(true);
	//viewer.getCamera()->getGraphicsContext()->getTraits()->vsync


    // get details on keyboard and mouse bindings used by the viewer.
    viewer.getUsage(*arguments.getApplicationUsage());

    

    // if user request help write it out to cout.
    if (arguments.read("-h") || arguments.read("--help"))
    {
        arguments.getApplicationUsage()->write(std::cout);
        return 1;
    }

    // any option left unread are converted into errors to write out later.
    arguments.reportRemainingOptionsAsUnrecognized();

    // report any errors if they have occured when parsing the program aguments.
    if (arguments.errors())
    {
        arguments.writeErrorMessages(std::cout);
        return 1;
    }
    
    if (arguments.argc()<=1)
    {
        arguments.getApplicationUsage()->write(std::cout,osg::ApplicationUsage::COMMAND_LINE_OPTION);
        return 1;
    }

    // extract the filenames from the arguments list.
    
    for(int pos=1;pos<arguments.argc();++pos)
    {
        if (arguments.isString(pos)) fileList.push_back(arguments[pos]);
    }

    if (fileList.size()<1)
    {
        return 1;
    }

    // set up the use of stereo by default.
    osg::DisplaySettings* ds = viewer.getDisplaySettings();
    if (!ds) ds = osg::DisplaySettings::instance();
    if (ds) ds->setStereo(true);

	
	viewer.setThreadingModel(osgViewer::ViewerBase::SingleThreaded);
    // realize the viewer
    viewer.realize();

	// init display
	osd.setColor(osg::Vec4d(0.45, 0.50, 0.7, 0.9));
    osd.setTextSize(24);
    osd.setText("SIV: press UP/DOWN to select Image");

    // creat the scene from the file list.

    rootNode = createScene(fileList,_radius,_height,_length);
	
	
	
	
    // set the scene to render
    root->addChild(&osd.getGroup());
    root->addChild(rootNode.get());
    root->setDataVariance(osg::Object::DYNAMIC);
	osd.setPosition(osg::Vec3d(10.0f,10.0f,0.0f));
    viewer.setSceneData(root);
	viewer.getCamera()->setClearColor( osg::Vec4( 0., 0., 0., 1. ) );
    viewer.getCamera()->setCullMask(0xffffffff);
    viewer.getCamera()->setCullMaskLeft(0x00000001);
    viewer.getCamera()->setCullMaskRight(0x00000002);
    
	osg::Timer timer;
	osg::Timer_t start_tick = timer.tick();
	
	
    while( !viewer.done() )
    {
    	if(slideshow){
    		osg::Timer_t end_tick = timer.tick();
    		if(timer.delta_s(start_tick,end_tick)>slideDelay){
    		file_up();
    		start_tick=end_tick;
    		}
    	}
        viewer.frame();        
    }
    

    return 0;
}
