/*
# (C) 2009 Jürgen Löb 
# jloeb@main-host.de
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
# See the Attribution-NonCommercial-ShareAlike 3.0 Unported for more details.
#
*/


#include <iostream>

#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <curses.h>

#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include <usb.h>
#include <math.h>
#include <libconfig.h++>
#include <typeinfo>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <getopt.h>

using namespace std;

struct TrackingData{
double viewmatrix[16];
double yaw;
double pitch;
double roll;
};

struct Vec3d {
        double x;
        double y;
        double z;
};



double zero_yaw=0;
double zero_pitch=0;
double zero_roll=0;

TrackingData trackingdata;


int socket_descriptor;
struct sockaddr_in address;

char* multicastAddress="224.0.0.42";
int multicastPort=4242;
int ratio=100; // cut off peaks

char* calibrationfile=".vr920";
libconfig::Config config;
int calmode=0;
int monitormode=0;
char key;
bool quit = false;

Vec3d acc_old=(Vec3d){0, 0, 0};
Vec3d mag_old=(Vec3d){0, 0, 0};

Vec3d acc_max = (Vec3d){0.1, 0.1, 0.1};
Vec3d acc_min = (Vec3d){-0.1, -0.1, -0.1};
Vec3d mag_max = (Vec3d){0.1, 0.1, 0.1};
Vec3d mag_min = (Vec3d){-0.1, -0.1, -0.1};


void help(){
	std::cout<<"vr920 [-h] [-?] [-m multicast ip] [-p multicast port] [-c calibrationfile]"<<std::endl<<std::endl;
	std::cout<<"(C) 2009 Jürgen Löb"<<std::endl;
	std::cout<<"jloeb(at)main-host.de"<<std::endl;
	}

void createMulticastSocket(){
   socket_descriptor = socket (AF_INET, SOCK_DGRAM, 0);
  if (socket_descriptor == -1) {
     perror ("socket()");
     exit (EXIT_FAILURE);
  }
  memset (&address, 0, sizeof (address));
  address.sin_family = AF_INET;
  address.sin_addr.s_addr = inet_addr (multicastAddress);
  address.sin_port = htons (multicastPort);

  printf("Sending multicasts to %s:%d.\n",multicastAddress,multicastPort );

}


void parsearguments(int argc, char *argv[]){
	int c;
	while ((c = getopt (argc, argv, "h?m:p:c:")) != -1)
         switch (c)
           {
           case 'h':
             help();
	     exit(0);
             break;
           case 'm':
             multicastAddress = optarg;
             break;
	   case 'p':
             multicastPort = atoi(optarg);
             break;
           case 'c':
             calibrationfile = optarg;
             break;
           case '?':
	     help();
             if (optopt == 'c')
               fprintf (stderr, "Option -%c requires an argument.\n", optopt);
             exit(0);
           default:
             return;
           }


	}

void readconfig(){
	try{
        config.readFile(calibrationfile);
        }
        catch(libconfig::ParseException e){
	std::cout << "Error parsing calibration file "<< calibrationfile << " !!!" << std::endl;
        std::cout << typeid(e).name() << std::endl;
        std::cout<< ((libconfig::ParseException)e).getError() << " in line " << ((libconfig::ParseException)e).getLine() << std::endl;
        }
        catch(libconfig::ConfigException e){

        std::cout << "Error loading calibration file "<< calibrationfile << " !!!" << std::endl;
        std::cout << typeid(e).name() << std::endl;    
        }

        if(!config.lookupValue("acc_max_x",acc_max.x)) std::cout << "Error loading acc_max_x from calibration file!!!" << std::endl;
	if(!config.lookupValue("acc_max_y",acc_max.y)) std::cout << "Error loading acc_max_y from calibration file!!!" << std::endl;
	if(!config.lookupValue("acc_max_z",acc_max.z)) std::cout << "Error loading acc_max_z from calibration file!!!" << std::endl;

	if(!config.lookupValue("acc_min_x",acc_min.x)) std::cout << "Error loading acc_min_x from calibration file!!!" << std::endl;
	if(!config.lookupValue("acc_min_y",acc_min.y)) std::cout << "Error loading acc_min_y from calibration file!!!" << std::endl;
	if(!config.lookupValue("acc_min_z",acc_min.z)) std::cout << "Error loading acc_min_z from calibration file!!!" << std::endl;

	if(!config.lookupValue("mag_max_x",mag_max.x)) std::cout << "Error loading acc_mag_x from calibration file!!!" << std::endl;
	if(!config.lookupValue("mag_max_y",mag_max.y)) std::cout << "Error loading mag_max_y from calibration file!!!" << std::endl;
	if(!config.lookupValue("mag_max_z",mag_max.z)) std::cout << "Error loading mag_max_z from calibration file!!!" << std::endl;

	if(!config.lookupValue("mag_min_x",mag_min.x)) std::cout << "Error loading mag_min_x from calibration file!!!" << std::endl;
	if(!config.lookupValue("mag_min_y",mag_min.y)) std::cout << "Error loading mag_min_y from calibration file!!!" << std::endl;
	if(!config.lookupValue("mag_min_z",mag_min.z)) std::cout << "Error loading mag_min_z from calibration file!!!" << std::endl;

	if(!config.lookupValue("zero_yaw",zero_yaw)) std::cout << "Error loading zero_pitch from calibration file!!!" << std::endl;
	if(!config.lookupValue("zero_pitch",zero_pitch)) std::cout << "Error loading zero_pitch from calibration file!!!" << std::endl;
	if(!config.lookupValue("zero_roll",zero_roll)) std::cout << "Error loading zero_roll from calibration file!!!" << std::endl;



	
}

void writeconfig(){
	libconfig::Setting &setting =config.getRoot ();	
	if(setting.exists("acc_max_x")) setting.remove("acc_max_x");
	setting.add("acc_max_x",libconfig::Setting::TypeFloat)=acc_max.x;
	if(setting.exists("acc_max_y")) setting.remove("acc_max_y");
	setting.add("acc_max_y",libconfig::Setting::TypeFloat)=acc_max.y;
	if(setting.exists("acc_max_z")) setting.remove("acc_max_z");
	setting.add("acc_max_z",libconfig::Setting::TypeFloat)=acc_max.z;
	if(setting.exists("acc_min_x")) setting.remove("acc_min_x");
	setting.add("acc_min_x",libconfig::Setting::TypeFloat)=acc_min.x;
	if(setting.exists("acc_min_y")) setting.remove("acc_min_y");
	setting.add("acc_min_y",libconfig::Setting::TypeFloat)=acc_min.y;
	if(setting.exists("acc_min_z")) setting.remove("acc_min_z");
	setting.add("acc_min_z",libconfig::Setting::TypeFloat)=acc_min.z;
	if(setting.exists("mag_max_x")) setting.remove("mag_max_x");
	setting.add("mag_max_x",libconfig::Setting::TypeFloat)=mag_max.x;
	if(setting.exists("mag_max_y")) setting.remove("mag_max_y");
	setting.add("mag_max_y",libconfig::Setting::TypeFloat)=mag_max.y;
	if(setting.exists("mag_max_z")) setting.remove("mag_max_z");
	setting.add("mag_max_z",libconfig::Setting::TypeFloat)=mag_max.z;
	if(setting.exists("mag_min_x")) setting.remove("mag_min_x");
	setting.add("mag_min_x",libconfig::Setting::TypeFloat)=mag_min.x;
	if(setting.exists("mag_min_y")) setting.remove("mag_min_y");
	setting.add("mag_min_y",libconfig::Setting::TypeFloat)=mag_min.y;
	if(setting.exists("mag_min_z")) setting.remove("mag_min_z");
	setting.add("mag_min_z",libconfig::Setting::TypeFloat)=mag_min.z;

	if(setting.exists("zero_yaw")) setting.remove("zero_yaw");
	setting.add("zero_yaw",libconfig::Setting::TypeFloat)=zero_yaw;
	if(setting.exists("zero_pitch")) setting.remove("zero_pitch");
	setting.add("zero_pitch",libconfig::Setting::TypeFloat)=zero_pitch;
	if(setting.exists("zero_roll")) setting.remove("zero_roll");
	setting.add("zero_roll",libconfig::Setting::TypeFloat)=zero_roll;

	
	try{
        config.writeFile(calibrationfile);
        }
        catch(libconfig::ParseException e){
        std::cout << "Error writing calibration file "<< calibrationfile << " !!!" << std::endl;
        std::cout << typeid(e).name() << std::endl;
        std::cout<< ((libconfig::ParseException)e).getError() << " in line " << ((libconfig::ParseException)e).getLine() << std::endl;

        }
        catch(libconfig::ConfigException e){
        std::cout << "Error writing calibration file "<< calibrationfile << " !!!" << std::endl;
        std::cout << typeid(e).name() << std::endl;     
        }

}

void zero(){
	zero_yaw=trackingdata.yaw+zero_yaw;
	zero_pitch=trackingdata.pitch+zero_pitch;
	zero_roll=trackingdata.roll+zero_roll;
}


// Signals handler

void catchQuit(int sig) {
	printf ("Caught signal %d Shutting down.\n",sig);
	quit = true;
}


// USB Stuff

struct usb_device *find_device(int vendor, int product) {
    struct usb_bus *bus;
    
    for (bus = usb_get_busses(); bus; bus = bus->next) {
		struct usb_device *dev;
		
		for (dev = bus->devices; dev; dev = dev->next) {
			if (dev->descriptor.idVendor == vendor
			&& dev->descriptor.idProduct == product)
			return dev;
		}
    }
    return NULL;
}




int main(int argc, char *argv[]) {
	if (argc>1) parsearguments(argc,argv);
	std::string filename(getenv("HOME"));
	if(calibrationfile[0]!='/'){		
		filename.append("/");
		filename.append(calibrationfile);
		calibrationfile=(char*)filename.c_str();
	}
	std::cout<<"Using calibrationfile:" << calibrationfile<<std::endl;

	readconfig();

	createMulticastSocket();

	
	// Setup signal handler
	
	signal(SIGTERM, catchQuit);
	signal(SIGQUIT, catchQuit);
	signal(SIGABRT, catchQuit);
	signal(SIGINT,  catchQuit);
	
	
	// Initialize the USB device.
	
    int ret, vendor, product;
    struct usb_device *dev;
	struct usb_dev_handle *devh;
    char buf[65535], *endptr;

    usb_init();
    usb_set_debug(255);
    usb_find_busses();
    usb_find_devices();

	vendor  = 0x1bae;
	product = 0x0002;

    dev = find_device(vendor, product);
	if(dev == NULL) {
		std::cout << "No VR920 attached to this system found." << endl;
		endwin();
		exit(1);
	}
	
    devh = usb_open(dev);

	if(!usb_get_driver_np(devh, 3, buf, sizeof(buf))) {
		printf( "Looks like the USB HID driver has claimed it. Detaching it.\n");
		ret = usb_detach_kernel_driver_np(devh, 3);
	}

	ret = usb_set_configuration(devh, 0x0000001);
		
	ret = usb_claim_interface(devh, 3);
	if (ret != 0) {
		usb_release_interface(devh, 3);
	        usb_close(devh);
		std::cout << "Couldn't claim interface: Error " << ret << endl;
	}
	
	
	// curses init
	initscr();
	cbreak();
   	noecho();
   	keypad(stdscr, TRUE);
   	nodelay(stdscr, TRUE);
   	scrollok(stdscr, TRUE); 
	raw();
	sleep(1);


	int send=0;

	while(!quit) {
		send=(send+1)%5;

		attron(A_REVERSE);
		if(!calmode) mvprintw(0,0,"VR90 - C-callibrate M-monitor data Q-quit.                                     \n" );
		else         mvprintw(0,0,"VR90 - C-normal mode R-reset calibration S-save calibration Z-set zero Q-quit. \n" );
		attroff(A_REVERSE);		

		key=getch();
		if(key!=ERR && key!=-1){
			if (*keyname(key)=='c') {
				calmode=(calmode+1)%2;
				if(!calmode) readconfig();
			}
			if(calmode){
				if (*keyname(key)=='s') writeconfig();
				if (*keyname(key)=='z') zero();
				if (*keyname(key)=='r') {
					acc_max = (Vec3d){0.1, 0.1, 0.1};
					acc_min = (Vec3d){-0.1, -0.1, -0.1};
					mag_max = (Vec3d){0.1, 0.1, 0.1};
					mag_min = (Vec3d){-0.1, -0.1, -0.1};
					zero_yaw=0;
					zero_pitch=0;
					zero_roll=0;
					}

			}
			else{
			if (*keyname(key)=='m') monitormode=(monitormode+1)%2;
			}
			if (*keyname(key)=='q') {quit=1;}	
}
		// Update the VR920 data
		
		
		ret = usb_interrupt_read(devh, 0x00000083, buf, 0x0000010, 1000);
		
		Vec3d acc;
		Vec3d mag;
		
		acc.x = double(*(short*)(buf + 3));
		acc.z = double(*(short*)(buf + 5));
		acc.y = double(*(short*)(buf + 7));
		mag.x = -double(*(short*)(buf + 9));
		mag.z = -double(*(short*)(buf + 11));
		mag.y = -double(*(short*)(buf + 13));


if(calmode){
//minmax		
		if(acc.x > acc_max.x) acc_max.x = acc.x;
		if(acc.x < acc_min.x) acc_min.x = acc.x;
		if(acc.y > acc_max.y) acc_max.y = acc.y;
		if(acc.y < acc_min.y) acc_min.y = acc.y;
		if(acc.z > acc_max.z) acc_max.z = acc.z;
		if(acc.z < acc_min.z) acc_min.z = acc.z;
			
		if(mag.x > mag_max.x) mag_max.x = mag.x;
		if(mag.x < mag_min.x) mag_min.x = mag.x;
		if(mag.y > mag_max.y) mag_max.y = mag.y;
		if(mag.y < mag_min.y) mag_min.y = mag.y;
		if(mag.z > mag_max.z) mag_max.z = mag.z;
		if(mag.z < mag_min.z) mag_min.z = mag.z;
		}
		
//normalize
		acc.x = ((acc.x-acc_min.x-(acc_max.x - acc_min.x)/2)/(acc_max.x - acc_min.x)*2);
		acc.y = ((acc.y-acc_min.y-(acc_max.y - acc_min.y)/2)/(acc_max.y - acc_min.y)*2);
		acc.z = ((acc.z-acc_min.z-(acc_max.z - acc_min.z)/2)/(acc_max.z - acc_min.z)*2);
		mag.x = ((mag.x-mag_min.x-(mag_max.x - mag_min.x)/2)/(mag_max.x - mag_min.x)*2);
		mag.y = ((mag.y-mag_min.y-(mag_max.y - mag_min.y)/2)/(mag_max.y - mag_min.y)*2);
		mag.z = ((mag.z-mag_min.z-(mag_max.z - mag_min.z)/2)/(mag_max.z - mag_min.z)*2);


if(calmode){
	     mvprintw(9,0, "accellerometer x:%f\ty:%f\tz:%f\n",acc.x,acc.y,acc.z);
	     mvprintw(10,0,"magnetometer   x:%f\ty:%f\tz:%f\n",mag.x,mag.y,mag.z);

	     mvprintw(12,0,"acc_min.x: %f\tacc_min.y: %f\tacc_min.z: %f",acc_min.x,acc_min.y,acc_min.z);
	     mvprintw(13,0,"acc_max.x: %f\tacc_max.y: %f\tacc_max.z: %f",acc_max.x,acc_max.y,acc_max.z);
	     mvprintw(14,0,"mag_min.x: %f\tmag_min.y: %f\tmag_min.z: %f",mag_min.x,mag_min.y,mag_min.z);
             mvprintw(15,0,"mag_max.x: %f\tmag_max.y: %f\tmag_max.z: %f",mag_max.x,mag_max.y,mag_max.z);

	     mvprintw(17,0,"During calibration make sure that the display of the device is displaying\nsomething.");
	     mvprintw(19,0,"Turn around the device until the min/max values don't change.");
	     mvprintw(21,0,"Press S to save callibration, R to reset, Z to set zero, C to exit\ncallibration mode");

//basic filter
		//cout<<fabs(fabs(acc.x)-fabs(acc_old.x))<<"<"<<(acc_max.x - acc_min.x)/ratio<<endl;
		}
		if(fabs(fabs(acc.x)-fabs(acc_old.x))>(acc_max.x - acc_min.x)/ratio) acc.x=acc_old.x;
                else{
			acc.x=(acc_old.x*39+acc.x)/40;
			acc_old.x=acc.x;
		}
		if(fabs(fabs(acc.y)-fabs(acc_old.y))>(acc_max.y - acc_min.y)/ratio) acc.y=acc_old.y;
                else{
			acc.y=(acc_old.y*39+acc.y)/40;
			acc_old.y=acc.y;
		}
		if(fabs(fabs(acc.z)-fabs(acc_old.z))>(acc_max.z - acc_min.z)/ratio) acc.z=acc_old.z;
                else{
			acc.z=(acc_old.z*39+acc.z)/40;
 			acc_old.z=acc.z;
		}
		if(fabs(fabs(mag.x)-fabs(mag_old.x))>(mag_max.x - mag_min.x)/ratio) mag.x=mag_old.x;
                else{
			mag.x=(mag_old.x*39+mag.x)/40;		
			mag_old.x=mag.x;
		}
		if(fabs(fabs(mag.y)-fabs(mag_old.y))>(mag_max.y - mag_min.y)/ratio) mag.y=mag_old.y;
                else{
			mag.y=(mag_old.y*39+mag.y)/40;		
			mag_old.y=mag.y;
		}
		if(fabs(fabs(mag.z)-fabs(mag_old.z))>(mag_max.z - mag_min.z)/ratio) mag.z=mag_old.z;
		else{
			mag.z=(mag_old.z*39+mag.z)/40;	
			mag_old.z=mag.z;
		}


		
double yaw;
double mag_pitch;
double mag_roll;
double acc_pitch;
double acc_roll;

acc_roll=atan2(acc.x, sqrt(acc.y*acc.y + acc.z*acc.z))*180/M_PI; 
//acc_pitch=atan2(acc.y, sqrt(acc.x*acc.x + acc.z*acc.z))*180/M_PI;
acc_pitch=atan2(acc.y, sqrt(acc.x*acc.x + acc.z*acc.z))*180/M_PI;


double xh= mag.x*cos((-acc_roll)/180.0*M_PI)+mag.y*sin(-acc_roll/180.0*M_PI)*sin((acc_pitch)/180.0*M_PI)-mag.z*sin(-acc_roll/180.0*M_PI)*cos((acc_pitch)/180.0*M_PI);
double yh= mag.y*cos(acc_pitch/180.0*M_PI)+mag.z*sin(acc_pitch/180.0*M_PI);

trackingdata.yaw=atan2(xh,yh)*180/M_PI-zero_yaw;
trackingdata.roll=acc_roll-zero_roll;
trackingdata.pitch=acc_pitch-zero_pitch;
		
		double A       = cos(-trackingdata.pitch/180.0*M_PI);
		double B       = sin(-trackingdata.pitch/180.0*M_PI);
    		double C       = cos(-trackingdata.roll/180.0*M_PI);
   		double D       = sin(-trackingdata.roll/180.0*M_PI);
   		double E       = cos(trackingdata.yaw/180.0*M_PI);
   		double F       = sin(trackingdata.yaw/180.0*M_PI);
		double AD      =   A * D;
    		double BD      =   B * D;
		
		trackingdata.viewmatrix[ 0] =  C * E;
		trackingdata.viewmatrix[ 1] = -C * F;
		trackingdata.viewmatrix[ 2] = D;
		trackingdata.viewmatrix[ 3] = 0;
		trackingdata.viewmatrix[ 4] = BD * E + A * F;
		trackingdata.viewmatrix[ 5] = -BD * F + A * E;
		trackingdata.viewmatrix[ 6] = -B * C;
		trackingdata.viewmatrix[ 7] = 0;
		trackingdata.viewmatrix[ 8] = -AD * E + B * F;
		trackingdata.viewmatrix[ 9] = AD * F + B * E;
		trackingdata.viewmatrix[10] =  A * C;
		trackingdata.viewmatrix[11] = 0;
		trackingdata.viewmatrix[12] = 0;
		trackingdata.viewmatrix[13] = 0;
		trackingdata.viewmatrix[14] = 0;
		trackingdata.viewmatrix[15] = 1;

		if(calmode||monitormode){
			mvprintw(3,0,"yaw: %f\n",trackingdata.yaw);
			mvprintw(4,0,"acc_roll: %f\n",trackingdata.roll);
			mvprintw(5,0,"acc_pitch: %f\n",trackingdata.pitch);
}
		
		if(!send){
			if (sendto( socket_descriptor,&trackingdata,sizeof(TrackingData),0,(struct sockaddr *) &address,sizeof (address)) < 0) {
       			perror ("sendto()");
       			quit=1;
    			}
		}

		refresh();
		usleep(1000); // Don't max out CPU
	}
	
	
	usb_release_interface(devh, 3);
	usb_close(devh);
	
	// shutdown socket
	shutdown(socket_descriptor, 2);
   	// close socket
   	close(socket_descriptor);

	
	//cleanup curses
	endwin();
	
	return 0;
}


